globals [
  ;; these are defined in the interface sliders
  ;; error-max    ;;  "errors" ( diversity or E ) will be on [-Error-max, Error-max],
                  ;;  with the largest range used being [-1,1]
  ;; population   ;;  number of people in the theater
  ;; quality      ;;  absolute quality of the show, ie Q
  ;; error-size   ;;
  ;; threshold    ;; threshold signal that causes agent to stand up if randomization is not turned on
                  ;; where signal = Quality + Error
  ;;
  ;;
  pass              ;; which pass, 1 or 2
  T                 ;; threshold for standing on pass 2, same for everyone
  ;; quality        ;; actual quality of the perforance ( ie Q )

  sitting-color     ;;
  standing-color    ;; stand up on pass one
  persuaded-color   ;; stand up on pass two
  percent-standing       ;; count of patrons standing at end of pass 1
  percent-standing-final ;; percent standing at end of pass 2
  percent-persuaded      ;; percent persuaded by peer-pressure in pass 2
  standing-ovation?      ;; true or false
  verbose?               ;; set to true for debugging
]

turtles-own [
  my-signal-threshold   ;;        this is an individual threshold for comparing to perceived quality ( S = Q + error )
  my-peer-pressure-threshold ;;    % of audience standing that will cause  me to stand on pass two, if I am not already standing from pass 1
  my-error       ;;  chosen from a random distribution from -E to E
  my-signal      ;;  after a play, compute this as actual-quality plus my-error, so ranges from -E to (100 + E)
  standing?      ;;  initially false
]

to setup
  clear-all
  set verbose? false
  set standing-ovation? false
  set sitting-color grey
  set standing-color green
  set persuaded-color yellow

  ;; we have a performance, set the quality
  if random-Q? [ set quality random 101 ]

  create-turtles population [ setxy random-xcor random-ycor  set size 3 set shape "person" set color blue]
  ask turtles [

    ;; errors are always randomized, but the error-max can be slid to zero
    set my-error uniform error-max

    set my-signal-threshold threshold ;;
    if random-T?  [set my-signal-threshold random 101 ]  ;; hmmm, question - does it make sense for both  S and T bo be randomized, and if so,
                                                         ;; is random-T chosen on U[0,100] or some other distribution centered on global threshold?

    set my-signal -999  ;;  Set after the show ends to Q + E

    set my-peer-pressure-threshold x-percent
    if random-X? [ set my-peer-pressure-threshold random 101 ]




    ;; they are sitting so
    set standing? false
    set color sitting-color
  ]
  print ( word population " patrons were seated.")
  reset-ticks ;;  ticks starts at 0 not 1
end

to-report uniform [eee]
     let ranger (2 * eee )
     let xc random (ranger + 1)  ;; x will be an integer between 0 and 2E
     set xc ( xc - eee )         ;; now x is between -E and E inclusive
     report xc
end


to go
  set pass ( ticks + 1 )     ;; because ticks start at 0
  if ( pass > 2 ) [ stop ]  ;; two passes is all that is needed to run the model

  if verbose? [ print (word "this is pass #" pass) ]
  if (ticks = 0) [
    print (word  "The show quality was " quality )
   ]  ;; the show happens

  if ( pass = 1 )
 [ ask turtles   [
    if verbose? [print ( word "Patron " who " is doing the first pass with show Q = " quality )]


       ;; compute my signal, that is, how I perceive the show
        ;;;;;;;;;;;;;;;;;;;;;;;;;set my-error uniform error-max

       set my-signal (quality + my-error) ;; the play's quality as I see it


    if verbose? [    print ( word ".... my-error = " my-error ", my-signal = " my-signal ", and my-signal-threshold = " my-signal-threshold)]

      ;; then I stand up if my-signal is over my-signal-threshold

       if-else ( my-signal > my-signal-threshold ) [
         if verbose? [  print "....standing up" ]
         set standing? true
         set color standing-color
        ]
        [
      if verbose? [ print ".... remaining sitting."]
    ]
    if verbose? [ print " " ]
  ]
    set percent-standing  ( 100 * count turtles with [ standing? = true ]) / ( count turtles )
    print (word "At the end of the first pass,  percent standing = " precision percent-standing 0)
    if verbose? [print " "]
  ]  ;; end of pass 1

   if (pass = 2 )
   [


    if verbose? [print (word "Starting second pass, with percent standing after passs 1 = " percent-standing)]
   ;; if ( percent-standing > X-percent )[ ask turtles with [standing? = false]  ;;  in version 1, this is wrong, tests global setting

   ask turtles with [standing? = false]  ;;  in version 2, this is better
      [
          if ( percent-standing > my-peer-pressure-threshold )[                      ;; this tests each individual person's threshhold
               set standing? true
               if verbose? [print (word "Patron " who " now stands up")]
               set color persuaded-color ]
      ]

    ;; ok now count again
    set percent-standing-final 100 * ( count turtles with [ standing? = true ]) / ( count turtles )
    set percent-persuaded 100 * count turtles with [color = persuaded-color] / (count turtles )
    if-else  percent-standing-final = 100 [ set standing-ovation? true ][ set standing-ovation? false]

    wrap-up-show
  ] ;; end of pass 2

  tick
end

to wrap-up-show
    print " ========================================="
    print ( word " The show quality was " quality )
    print ( word " At the end of the first pass,  percent standing  : " precision percent-standing 0)
    print ( word " Then, this percent was also persuaded to stand up: " precision percent-persuaded 0)
    print ( word " So, the final total percent standing             : " precision percent-standing-final 0)
    if-else ( standing-ovation?  )
          [ print "   Standing ovation!"]
          [ print "   Did not get a full standing ovation"]

  ;;  print " ========================================="
  ;;  print " update graphs and output here "

  let show-text (word "For Q=" quality ", T=" threshold ", E="  error-max ", X=" X-percent ", %pass-1=" precision percent-standing 0 ", %pass-2="  precision percent-persuaded 0 " %total=" precision percent-standing-final 0)                                   ;;////////////////////////////////////////////////////////////////////////////
  print show-text
  ;;  print " ========================================="

  if population = 1 [ display-visual ]

end
to display-visual ;; for only one turtle


  let my-T 0
  let my-E 0
  let my-S 0
  let my-Q quality

  ask turtle 0 [ set hidden? true
  set my-T my-signal-threshold
  set my-E my-error
  set my-S my-signal
  ]

  analyze-one-Emax error-max my-Q   my-T my-S  ;; this is the one to come home to

end

to analyze-one-Emax [ amax my-Q   my-T my-S ]

  ask patches [ set pcolor gray]
  ask patches with [ pxcor > -10 and pxcor < 10 and pycor < 3 and pycor > -3 ] [ set pcolor white ] ;; 21 patches wide, each about 5 points
  let white-count count patches with [ pcolor = white ]

  ;; make a vertical black bar at T
  let plot-T ( my-T / 5)  - 10
  ask patches with [ pxcor = round plot-T and pycor < 3 and pycor > -3 ] [ set pcolor black ]

  ;; make a blue short vertical bar at Q
  let plot-Q ( my-Q / 5) - 10 ;
  ask patches with [pxcor = round plot-Q and pycor > -1 and pycor < 6 ] [ set pcolor blue ]

  ;; make a red short vertical bar at S ( what the dice actually rolled )
  let plot-S ( my-S / 5) - 10 ;
  ask patches with [ pxcor = round plot-S and pycor < 3 and pycor > 0 ] [ set pcolor red ]


  ;; clear out old blue bar
  ;; ask patches with [  pycor = 3 ]  [ set pcolor white]

  ;; make a new blue horizontal bar showing what the dice MIGHT have rolled ( ie, Q - E to Q + E)
  let lefter  (my-Q - amax)
  let righter (my-Q + amax)
  set lefter  round ( ( lefter  / 5 ) - 10);
  set righter round ( ( righter / 5 ) - 10);
  ask patches with [pxcor >= lefter and pxcor <= righter and pycor = 3 ]  [ set pcolor blue]

  ;; mark possible places the red bar could land that would result in standing up on the first pass
  ask patches with [pxcor > plot-T and pxcor <= righter   and pycor < 3 and pycor > -3 ][ set pcolor green]

  ;; compute the odds of standing with these settings ( Q and T fixed,  S has uniform distribution around Q of half-width E )
  let green-count count patches with [ pcolor = green and pycor = 0 ]  ;; count values of S that result in standing
  let blue-count  count patches with [ pcolor = blue  and pycor = 3 ]  ;; count possible values S could take on
  let stand-pct ( 100 * green-count / blue-count)

  ;; BUG -- truncated pct at 100 max instead of locating the bug in my code!  ////////////////////////////////////////////////////

  if (stand-pct > 100 ) [ set stand-pct 100 ]
  print (word "Given E = " amax  " ,  Q = " my-Q " , T= " my-T ",  the odds of standing in pass 1 are " round stand-pct " percent" )

  ;; and clean up the display out of order to put the red on top of the green
  ask patches with [ pxcor = round plot-S and pycor < 3 and pycor > 0 ] [ set pcolor red ]

end
@#$#@#$#@
GRAPHICS-WINDOW
250
15
687
453
-1
-1
13.0
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
20
105
83
138
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
85
105
160
138
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
165
105
228
138
run
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
20
155
192
188
quality
quality
0
100
40.0
5
1
NIL
HORIZONTAL

SLIDER
20
235
192
268
threshold
threshold
0
100
40.0
5
1
NIL
HORIZONTAL

SLIDER
20
65
192
98
population
population
1
100
49.0
1
1
NIL
HORIZONTAL

SLIDER
20
395
192
428
X-percent
X-percent
0
100
55.0
5
1
NIL
HORIZONTAL

SLIDER
20
315
192
348
Error-max
Error-max
0
100
20.0
5
1
NIL
HORIZONTAL

SWITCH
20
185
137
218
random-Q?
random-Q?
1
1
-1000

SWITCH
20
265
132
298
random-T?
random-T?
1
1
-1000

SWITCH
20
425
132
458
random-X?
random-X?
0
1
-1000

TEXTBOX
25
20
240
61
Set the pop. to 1 then click SETUP and RUN to make the slider diagram
12
0.0
0

TEXTBOX
20
355
170
395
The errors are always randomized within +/- E
12
0.0
1

@#$#@#$#@
## WHAT IS IT?

This is a very simple implementation of the SOP ( Standing Ovation Problem) that Scott E. Page uses in his Model Thinking on-line class. A paper describing it is linked below.

The situation modeled is this:  Patrons visit a live performance at the theater. At the end of the performance, each patron makes a decision to stand up or not.  The question is, will a "standing ovation" occur -- that is, will everyone stand?

Version 1 of this model had a bug which caused all patrons to have the same threshold for standing in pass-2 so either no one stood or everyone stood.  That has been fixed in this version.


## HOW IT WORKS

This model uses two steps to see if a standing ovation occurs. 

On "setup", the patrons enter the theater and are seated in random locations.  Then a new performance occurs, with absolute quality Q.     

Each patron has a somewhat different concept of how good they personally thought the performance was, modeled as a signal S, computed as S = Q + error,  where errors are randomly selected from a uniform distribution spanning from -E to +E.  

Each patron has a somewhat different threshold T for how good a performance has to be for them to stand up on their own.  Those who perceive S > T stand up.  That ends step 1.

Then, in step 2,   everyone looks around to see how many people also stood up.  If that percent is above their personal peer-pressure threshold X, even if they originally decided to stay seated, they will now also stand up. In this version of the SOP,  no one ever elects to sit down again if very few other people also stood up in step 1.


## HOW TO USE IT

The user uses the slider to set the number of patrons attending, possibly changes other parameters using the sliders and switches, or not,  and clicks SETUP.   The patrons are shown in the theater, color-coded grey to indicate they are seated.    The count of patrons is printed at the bottom, in the "Command Center".

Then the user clicks the STEP button,  and the model computes who will stand up right away for these settings of the parameters.   The show Quality is computed, and the Quality and number standing after pass 1 is printed in the Command Center.  Those who stood in this pass are now color-coded green.

Then the user clicks the STEP button one more time.  The model computes who will stand on the second pass, and color-codes them yellow.  Summary counts for who stood on each pass are printed at the bottom, along with an evaluation of whether a standing ovation occurred or not.

If the user prefers,  instead of clicking STEP twice, they can click RUN once and it will run both steps and then stop.

After the model has completed, hitting STEP or RUN has no effect.  The SETUP button must be clicked again to reset the model.


IMPORTANT NOTE: 

* The parameters have to be set BEFORE the SETUP button is clicked, or they won't have an effect for that run.


## THE BAR-CHART DISPLAY ( HIDDEN "EASTER EGG")

There is a second hidden feature in the model.  If you set the size of the population to 1,  then click SETUP and RUN, what you get displayed instead is a bar-chart illustrating what the calculations are looking at. It has no labels (sorry).

The absolute quality Q could vary from zero at the left of the white bar to 100 at the right, and the actual value randomly selected for it on this particular run is shown by the vertical blue bar.

The quality perceived by the users is bounded by the blue horizontal bar, centered on the blue vertical bar.   For high levels of E, the blue bar will be wide. For low levels of E, it will be narrow. For E = 0, all users perceive quality level Q. 

The threshold of perceived quality for the second pass ( peer-pressure ) is shown as a vertical black bar.  

The combination of factors representing standing patrons is shown in green.  That is the green area shows how many people will be standing, at the end of the second pass, given the parameters the user picked befor hitting SETUP.  

The percent of patrons standing will be the width of the green area divided by the total width of the blue horizontal line, capped at 100%.  

The values for Q, E, T, and percent standing, statistically,  after pass 1 are printed in the Command Center.   note: Computations for the end of pass 2 are not shown.

The value of S,  that is Q + error, that was randomly selected given the parameters chose, for this one patron is shown as a red vertical bar.  That number does not affect the computed percent that would be standing, statisticall -- it's just to confirm that the model is doing what we wanted it to do.


## INTERFACE CONTROLS


SETUP, STEP, and RUN:

The model can be run by simply clicking SETUP and then RUN.

To stop mid-stream after pass 1 to check things out, you can clilck SETUP then STEP, and click STEP a second time to complete the run.

Sliders set the base values for the controllable parameters, and little switches next to the sliders control whether the model will add random variation to that base value or not.  For a completely deterministic model, set all the switches OFF.

The sliders are as follows:

POPULATION:     sets how many people will attend the show.

QUALITY:        sets the absolute quality Q of the performance

THRESHOLD:      sets the threshold T for comparison to perceived quality ( Q + error)
		to be compared to.  If perceived quality exceeds T, the person stands
		in pass 1.  

ERROR-MAX:      sets the size of variation users might have around Q. 
                Variation is always pulled from a uniform distribution
		in the range -E to E

X-%:            Sets a second threshold -- for what percent of people need to be  
                standing at the end of the first pass for a person still seated to
                then stand up on the second pass.



## THINGS TO NOTICE

* If you set the population to 1, you get a bar-chart instead of a picture of people.


## THINGS TO TRY

* If you set the population to 1, you get a bar-chart instead of a picture of people.

* If you shut off all the randomizing switches, with a population of 1, and set Error-max to zero ( no diversity ), confirm that if Q is less than T, there is no green area  ( no one stands at all) and if Q is greater than T,  everyone stands on the first pass.

*  If Quality is below Threshold, but there is diversity ( E > 0 ) you can still get mamy people standing.    Try Q=45 and T=50 and vary the diversity E from zero to 100, and see what happens to the percent standing.  
 
## EXTENDING THE MODEL

See the paper cited below by Miller and Page for many directions to extend the model.

People might sit in rows, and only be influenced by people immediately around them, or in a cone of vision in frot of them.   People might have another threshold controlling whether they will sit down again if fewer than that percent of people are standing at the end of pass 1.  

## NETLOGO FEATURES

No special features were used in this model.

## RELATED MODELS

A model named "Standing ovation problem replica" was uploaded to the Modeling Commons 5 years ago by Jean van Haperen.  It's written in NetLogo 5.0.5.

## CREDITS AND REFERENCES

Scott E. Page's coursera course on Model Thinking
https://www.coursera.org/learn/model-thinking

The Standing Ovation Problem, by John H. Miller and Scott E. Page, April 13, 2004
http://www2.econ.iastate.edu/tesfatsi/StandingOvation.MillerPage.pdf

Scott E. Page research on Diversity
https://sites.lsa.umich.edu/scottepage/research-2/diversity-research/

Author:  Version 1 of this model was developed by R. Wade Schuette,  uploaded 23-Nov-2019

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment-1-sanity-check" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>set percent-standing precision percent-standing 0
set percent-persuaded precision percent-persuaded 0
set percent-standing-final precision percent-standing-final 0</final>
    <timeLimit steps="2"/>
    <metric>percent-standing</metric>
    <metric>percent-persuaded</metric>
    <metric>percent-standing-final</metric>
    <metric>standing-ovation?</metric>
    <enumeratedValueSet variable="random-X?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-Q?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="X-percent">
      <value value="55"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Error-max">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population">
      <value value="49"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-T?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quality">
      <value value="41"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="threshold">
      <value value="40"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment-2-sweep-Q" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="2"/>
    <metric>percent-standing</metric>
    <metric>percent-persuaded</metric>
    <metric>percent-standing-final</metric>
    <metric>standing-ovation?</metric>
    <enumeratedValueSet variable="random-X?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-Q?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="X-percent">
      <value value="55"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Error-max">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population">
      <value value="49"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-T?">
      <value value="false"/>
    </enumeratedValueSet>
    <steppedValueSet variable="quality" first="0" step="5" last="100"/>
    <enumeratedValueSet variable="threshold">
      <value value="40"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment-3-sweep-T" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="2"/>
    <metric>percent-standing</metric>
    <metric>percent-persuaded</metric>
    <metric>percent-standing-final</metric>
    <metric>standing-ovation?</metric>
    <enumeratedValueSet variable="random-X?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-Q?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="X-percent">
      <value value="55"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Error-max">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population">
      <value value="49"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-T?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quality">
      <value value="41"/>
    </enumeratedValueSet>
    <steppedValueSet variable="threshold" first="0" step="5" last="100"/>
  </experiment>
  <experiment name="experiment-4-sweep-E" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="2"/>
    <metric>percent-standing</metric>
    <metric>percent-persuaded</metric>
    <metric>percent-standing-final</metric>
    <metric>standing-ovation?</metric>
    <enumeratedValueSet variable="random-X?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-Q?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="X-percent">
      <value value="55"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Error-max" first="0" step="5" last="100"/>
    <enumeratedValueSet variable="population">
      <value value="49"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-T?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quality">
      <value value="41"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="threshold">
      <value value="40"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment-5-sweep-E-T" repetitions="1" sequentialRunOrder="false" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="2"/>
    <metric>percent-standing</metric>
    <metric>percent-persuaded</metric>
    <metric>percent-standing-final</metric>
    <metric>standing-ovation?</metric>
    <enumeratedValueSet variable="random-X?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-Q?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="X-percent">
      <value value="55"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Error-max" first="0" step="5" last="100"/>
    <enumeratedValueSet variable="population">
      <value value="49"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-T?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quality">
      <value value="40"/>
    </enumeratedValueSet>
    <steppedValueSet variable="threshold" first="0" step="5" last="100"/>
  </experiment>
  <experiment name="experiment-6-sweep-E-T-sorted" repetitions="20" sequentialRunOrder="false" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="2"/>
    <metric>percent-standing</metric>
    <metric>percent-persuaded</metric>
    <metric>percent-standing-final</metric>
    <metric>standing-ovation?</metric>
    <enumeratedValueSet variable="random-X?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-Q?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="X-percent">
      <value value="55"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Error-max" first="0" step="5" last="100"/>
    <enumeratedValueSet variable="population">
      <value value="49"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-T?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quality">
      <value value="40"/>
    </enumeratedValueSet>
    <steppedValueSet variable="threshold" first="0" step="5" last="100"/>
  </experiment>
  <experiment name="experiment-7-sweep-E-T-4-cores" repetitions="20" sequentialRunOrder="false" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="2"/>
    <metric>percent-standing</metric>
    <metric>percent-persuaded</metric>
    <metric>percent-standing-final</metric>
    <metric>standing-ovation?</metric>
    <enumeratedValueSet variable="random-X?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-Q?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="X-percent">
      <value value="55"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Error-max" first="0" step="5" last="100"/>
    <enumeratedValueSet variable="population">
      <value value="49"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random-T?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="quality">
      <value value="40"/>
    </enumeratedValueSet>
    <steppedValueSet variable="threshold" first="0" step="5" last="100"/>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
