globals [
  tick-advance-amount               ; how much we advance the tick counter this time through
  max-tick-advance-amount           ; the largest tick-advance-amount is allowed to be
  init-avg-speed init-avg-energy    ; initial averages
  avg-speed avg-energy              ; current average
  avg-temp-type-1
   avg-temp-type-2
   avg-temp-type-3

  avg-energy-gray
  particle-size
  toggle-red-state
  toggle-green-state
  min-particle-energy
  max-particle-energy
  gas-particles-to-add
  initial-gas-temperature

  #-3-wall-particles
  #-2-wall-particles
  #-1-wall-particles

  #-3-wall-total-te
  #-2-wall-total-te
  #-1-wall-total-te

  init-temp#1
  init-temp#2
  init-temp#3

  left-piece-label
  middle-piece-label
  right-piece-label
  mouse-interaction
  kelvin-celcius-offset
  a
  b

  left-solid-length
  left-solid-width
  middle-solid-length
  middle-solid-width
  right-solid-length
  right-solid-width
]

breed [ gas-particles particle ]
breed [ walls wall ]
breed [ flashes flash ]
breed [ erasers eraser ]
breed [ arrowheads arrowhead ]
breed [ boundaries boundary ]

erasers-own [ pressure? ]
flashes-own [ birthday ]

gas-particles-own [
  speed mass energy          ; gas-particles info
  last-collision
  color-type
]

walls-own [
  energy
  valve-1?
  valve-2?
  pressure?
  my-pxcor
  my-pycor
  my-heading
  distance-advance
  color-type
  particle-type
  init-yoffset
]

to setup
  clear-all
  reset-ticks
  set particle-size 1.0
  set max-tick-advance-amount 0.02

  set #-3-wall-particles 0
  set #-2-wall-particles 0
  set #-1-wall-particles 0
  set kelvin-celcius-offset 273
  set gas-particles-to-add 2
  set mouse-interaction "none"
  set a 200
  set b .75
 ; set initial-gas-temperature 500
  set   left-solid-length 8
  set   left-solid-width 8
  set   middle-solid-length 2
  set   middle-solid-width 8
  set   right-solid-length 8
  set   right-solid-width 8

  set init-temp#1 left.start.temp ;+ kelvin-celcius-offset
  set init-temp#2 middle.start.temp ;+ kelvin-celcius-offset
  set init-temp#3 right.start.temp; + kelvin-celcius-offset

  set-default-shape flashes "square2"
  set-default-shape walls "circle"
  set-default-shape erasers "eraser"
  set-default-shape arrowheads "default"

  set min-particle-energy 100
  set max-particle-energy 500  ;(.5 ) * ( max-dist-in-tick-advance-amount  / max-tick-advance-amount ) ^ 2

  create-erasers 1 [ set hidden? true set size 1.2 set color [255 255 255 100] ]

  make-box

  ask walls [transfer-energy-mytype]

  ask walls [recolorshade-walls]

  ask gas-particles [ apply-speed-visualization ]

  set init-avg-speed avg-speed
  set init-avg-energy avg-energy

  update-variables
  do-plots
end


to go
   if end-simulation? [ stop ]
  ask walls [transfer-energy]
  ask walls [calculate-distance-advance-amount]

  ask walls [apply-vibration-motion-visualization]
  ask gas-particles [ apply-speed-visualization ]
  ask walls [recolorshade-walls]

  update-variables
  do-plots

  calculate-tick-advance-amount
  tick-advance tick-advance-amount

  display
end


to calculate-distance-advance-amount
   set distance-advance (distance-advance + (tick-advance-amount * (energy) * 9 ))
end


to-report end-simulation?
  let value-to-report false
  let auto-stop-simulation? ""
  if auto-stop-simulation? = "at time = 5" and ticks >= 5    [set value-to-report true]
  if auto-stop-simulation? = "at time = 10" and ticks >= 10  [set value-to-report true]
  if auto-stop-simulation? = "at time = 15" and ticks >= 15  [set value-to-report true]
  if auto-stop-simulation? = "at time = 20" and ticks >= 20  [set value-to-report true]
 ; set value-to-report true
  report value-to-report
end

to update-variables

  if any? gas-particles [
    set avg-speed  mean [ speed ] of gas-particles
    set avg-energy( mean [ energy ] of gas-particles - kelvin-celcius-offset)
  ]
  let wall-particles-type-1 walls with [particle-type = 1]
  let wall-particles-type-2 walls with [particle-type = 2]
  let wall-particles-type-3 walls with [particle-type = 3]



  ;show count gas-particles-type-1
  ifelse any? wall-particles-type-1 [
     set #-1-wall-particles count wall-particles-type-1
     set avg-temp-type-1 ( (mean [ energy ] of wall-particles-type-1)  - kelvin-celcius-offset)
     set #-1-wall-total-te ((mean [ energy ] of wall-particles-type-1)  * #-1-wall-particles )
     set left-piece-label avg-temp-type-1
   ; show "yes"
  ]
  [set left-piece-label "N/A"]
  ifelse any? wall-particles-type-2 [
     set #-2-wall-particles count wall-particles-type-2
     set avg-temp-type-2 ( (mean [ energy ] of wall-particles-type-2)  - kelvin-celcius-offset)
     set #-2-wall-total-te ((mean [ energy ] of wall-particles-type-2)  * #-2-wall-particles )
     set middle-piece-label avg-temp-type-2
   ; show "yes"
  ]
  [set right-piece-label "N/A"]
    ifelse any? wall-particles-type-3 [
     set #-3-wall-particles count wall-particles-type-3
     set avg-temp-type-3 ( (mean [ energy ] of wall-particles-type-3)  - kelvin-celcius-offset)
     set #-3-wall-total-te ((mean [ energy ] of wall-particles-type-3) * #-3-wall-particles )
     set right-piece-label avg-temp-type-3
   ; show "yes"
  ]
  [set right-piece-label "N/A"]

end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;GAS MOLECULES MOVEMENT;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to bounce  ; gas-particles procedure
  ; get the coordinates of the patch we'll be on if we go forward 1
  let bounce-patch nobody
  let bounce-patches nobody
  let hit-angle 0
  let this-patch patch-here
  let new-px 0
  let new-py 0
  let visible-wall nobody
  let particle-energy energy

  set bounce-patch  min-one-of walls in-cone ((sqrt (2)) / 2) 180 with [ myself != this-patch ] [ distance myself ]

  if bounce-patch != nobody [
    let wall-energy [energy] of bounce-patch
    let energy-difference (particle-energy - wall-energy)
    set new-px [ pxcor ] of bounce-patch
    set new-py [ pycor ] of bounce-patch
    set visible-wall walls-on bounce-patch

    if any? visible-wall with [ not hidden? ]  [
      set hit-angle towards bounce-patch
      ifelse (hit-angle <= 135 and hit-angle >= 45) or (hit-angle <= 315 and hit-angle >= 225) [
        set heading (- heading)
      ][
        set heading (180 - heading)
      ]
      set energy (energy - (energy-difference / 4))
      set speed speed-from-energy
      ask bounce-patch [set energy (energy + (energy-difference / 4)) ]
      if true [; show-wall-hits? [
        ask patch new-px new-py [
          sprout 1 [
            set breed flashes
            set color gray - 2
            set birthday ticks
          ]
        ]
      ]
    ]
  ]
end


to rewind-to-bounce  ; gas-particles procedure
  ; attempts to deal with particle penetration by rewinding the particle path back to a point
  ; where it is about to hit a wall
  ; the particle path is reversed 49% of the previous tick-advance-amount it made,
  ; then particle collision with the wall is detected again.
  ; and the particle bounces off the wall using the remaining 51% of the tick-advance-amount.
  ; this use of slightly more of the tick-advance-amount for forward motion off the wall, helps
  ; insure the particle doesn't get stuck inside the wall on the bounce.

  let bounce-patch nobody
  let bounce-patches nobody
  let hit-angle 0
  let this-patch nobody
  let new-px 0
  let new-py 0
  let visible-wall nobody

  bk (speed) * tick-advance-amount * .49
  set this-patch  patch-here

  set bounce-patch  min-one-of walls in-cone ((sqrt (2)) / 2) 180 with [ self != this-patch ] [ distance myself ]

  if bounce-patch != nobody [

    set new-px [pxcor] of bounce-patch
    set new-py [pycor] of bounce-patch
    set visible-wall walls-on bounce-patch

    if any? visible-wall with [not hidden?] [
      set hit-angle towards bounce-patch

      ifelse (hit-angle <= 135 and hit-angle >= 45) or (hit-angle <= 315 and hit-angle >= 225) [
        set heading (- heading)
      ][
        set heading (180 - heading)
      ]

      if true [;[show-wall-hits? [
        ask patch new-px new-py [
          sprout 1 [
            set breed flashes
            set color gray - 2
            set birthday ticks
          ]
        ]
      ]
    ]
  ]
  fd (speed) * tick-advance-amount * .51
end

to move  ; gas-particles procedure
  if patch-ahead (speed * tick-advance-amount) != patch-here [ set last-collision nobody ]
  jump (speed * tick-advance-amount)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;GAS MOLECULES COLLISIONS;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;from GasLab

to calculate-tick-advance-amount
  set tick-advance-amount max-tick-advance-amount

end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  mouse interaction procedures
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to mouse-action
 let snap-xcor 0
 let snap-ycor 0
 let orig-xcor 0
 let orig-ycor 0
 let eraser-window-walls nobody
 let eraser-window-gas-particles nobody

  ifelse mouse-down? [
    set orig-xcor mouse-xcor
    set orig-ycor mouse-ycor
    set snap-xcor round orig-xcor
    set snap-ycor round orig-ycor

    ask patches with [ pxcor = snap-xcor and pycor = snap-ycor ] [
      set eraser-window-walls  walls-here
     ; show eraser-window-walls
      set eraser-window-gas-particles gas-particles-here



      if mouse-interaction = "speed up gas-particles"  [

        ask erasers [
          set hidden? false
          set shape "spray paint"
          setxy orig-xcor orig-ycor
        ]
        ask eraser-window-gas-particles [
          set energy (energy * 1.1)
          set energy limited-particle-energy
          set speed speed-from-energy
        ;  apply-vibration-color-visualization
        ]
        ask eraser-window-walls [
          set energy (energy * 1.1)
          set energy limited-particle-energy
         ; apply-vibration-color-visualization
        ]



      ]

      if mouse-interaction = "slow down gas-particles" [
        ask erasers [
          set hidden? false
          set shape "spray paint"
          setxy orig-xcor orig-ycor
        ]
        ask eraser-window-gas-particles [
          set energy (energy / 1.1)
          set energy limited-particle-energy
          set speed speed-from-energy
          apply-speed-visualization
        ]
        ask eraser-window-walls [
          set energy (energy / 1.1)
          set energy limited-particle-energy
        ;  apply-vibration-color-visualization
        ]
      ]
    ]
    ask gas-particles with [ any? walls-here ] [ remove-from-walls ] ; deal with any walls drawn on top of gas-particles
  ][
    ask erasers [ set hidden? true ]
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; thermal energy transfer
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to transfer-energy
  let neighboring-walls walls-on neighbors4
  let my-energy energy
  let total-energy-from-neighbors 0

  if any? neighboring-walls [

  ask neighboring-walls [
    let energy-from-me (((energy) * (0.25)))
    set energy (energy - energy-from-me)
    set total-energy-from-neighbors total-energy-from-neighbors + energy-from-me
  ]

  set energy energy + total-energy-from-neighbors
  ]
end


to transfer-energy-mytype
  let my-type particle-type
  let neighboring-walls-all walls-on neighbors4
  let neighboring-walls neighboring-walls-all with [particle-type = my-type]
  let my-energy energy
  let total-energy-from-neighbors 0

  if any? neighboring-walls [

  ask neighboring-walls [
    let energy-from-me (((energy) * (0.25)))
    set energy (energy - energy-from-me)
    set total-energy-from-neighbors total-energy-from-neighbors + energy-from-me
  ]

  set energy energy + total-energy-from-neighbors
  ]
end


to check-slide-up
  if particle-type = 1 [set ycor slide-left-solid-up + init-yoffset]
  if particle-type = 2 [set ycor slide-middle-solid-up + init-yoffset]
  if particle-type = 3 [set ycor slide-right-solid-up + init-yoffset]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; particle speed and flash visualization procedures
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



to apply-vibration-motion-visualization
  set xcor my-pxcor
  check-slide-up
 ; set my-heading my-heading + (energy / 100) * (random-float 5  - random-float 5) ; (energy / 100)  - random-float (energy / 100)
;  set heading my-heading
  let displacement (0.05 * ((energy / a) ^ b) *  sin (distance-advance) )
  fd displacement
end




to apply-speed-visualization
  recolorshade-gas-particles
 ; if visualize-speed? = "arrows" [ scale-arrowheads ]
 ; if visualize-speed? = "different shades" [ recolorshade-gas-particles ]
 ; if visualize-speed? = "none" [ recolornone ]
end

to color-particle-and-link
  let this-link my-out-links
  let this-color-type color-type
  set color this-color-type
  ask this-link [ set color this-color-type ]
end

to scale-arrowheads
  let this-xcor xcor
  let this-ycor ycor
  let this-speed speed
  let this-heading heading
  let this-arrowhead out-link-neighbors
  let this-link my-out-links
  ask this-link [ set hidden? false ]
  ask this-arrowhead [
    set xcor this-xcor
    set ycor this-ycor
    set heading this-heading
    fd .5 + this-speed / 3
  ]
end

to-report my-color-type
  let this-color-type color-type
  report this-color-type
end


to recolorshade-gas-particles
  let this-link my-out-links
  let this-arrowhead out-link-neighbors

  let local-color-type my-color-type
  let energy-difference (energy)
  let r-value ((4.9) * (1 - 0.999 ^ (energy)))


  let this-color ((my-color-type + 4.99) - r-value)

  set color this-color
  ask this-link [ set color this-color ]
  ask this-arrowhead [ set color this-color ]
end

to recolornone
  let this-link my-out-links
  let this-arrowhead out-link-neighbors
  let this-color color-type
  set color color-type
  ask this-link [set color this-color]
  ask this-arrowhead [ set color this-color ]

end




to recolorshade-walls
  let this-link my-out-links
  let this-arrowhead out-link-neighbors

  let local-color-type my-color-type
  let energy-difference (energy)
  let r-value ((8.9) * (1 - 0.999 ^ (energy)))

  let this-color ((my-color-type + 4.99) - r-value)

  set color this-color
  ask this-link [ set color this-color ]
  ask this-arrowhead [ set color this-color ]
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  initialization procedures
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to make-box
  let vertical-line min-pycor
  let vertical-line-xcor 0.5
  let vertical-line-offset 8

  create-boundaries 1 [
    set xcor (vertical-line-xcor)  set shape "new-line" set color [255 255 255 120] set size 17
  ]
  create-boundaries 1 [
    set xcor (vertical-line-xcor + left-solid-width) set shape "new-line" set color [255 255 255 120] set size 17
  ]
 create-boundaries 1 [
    set xcor (vertical-line-xcor + left-solid-width + middle-solid-width) set shape "new-line" set color [255 255 255 120] set size 17
  ]

 create-boundaries 1 [
    set xcor (vertical-line-xcor + left-solid-width + middle-solid-width + right-solid-width) set shape "new-line" set color [255 255 255 120] set size 17
  ]


  ask patches with [ ((pycor >= (min-pycor + slide-left-solid-up)) and (pycor < (min-pycor + slide-left-solid-up + left-solid-length )) )
      and (pxcor > min-pxcor ) and (pxcor <= (left-solid-width)) ] [
      sprout 1 [set breed walls set color-type red set size 1.0 set particle-type 1 set shape "rounded-square" initialize-this-wall set init-yoffset( pycor - slide-left-solid-up)]
  ]

  ask patches with [ ((pycor >= (min-pycor + slide-middle-solid-up )) and (pycor < (min-pycor + slide-middle-solid-up + middle-solid-length )) )
      and (pxcor > (min-pxcor + left-solid-width )  and (pxcor <=  (min-pxcor + left-solid-width  + middle-solid-width))) ] [
      sprout 1 [set breed walls set color-type red set particle-type 2 set size 1.0 set shape "rounded-square" initialize-this-wall set init-yoffset( pycor - slide-middle-solid-up) ]
  ]

  ask patches with [ ((pycor >= (min-pycor + slide-right-solid-up )) and (pycor < (min-pycor + slide-right-solid-up + right-solid-length )) )
      and (pxcor > (min-pxcor + left-solid-width + middle-solid-width))  and (pxcor <= (min-pxcor  + middle-solid-width + left-solid-width + right-solid-width)) ] [
      sprout 1 [set breed walls set color-type red set size 1.0  set particle-type 3 set shape "rounded-square" initialize-this-wall set init-yoffset( pycor - slide-right-solid-up)]
  ]

end

to initialize-this-wall
  set valve-1? false
  set valve-2? false
  set pressure? false
  if particle-type =  1 [set energy left.start.temp + kelvin-celcius-offset]
  if particle-type =  2 [set energy middle.start.temp  + kelvin-celcius-offset]
  if particle-type =  3 [set energy right.start.temp  + kelvin-celcius-offset]
  set my-pxcor pxcor
  set my-pycor pycor
  set my-heading random 360
  set distance-advance 0

end


to setup-gas-particles  ; gas-particles procedure
  set shape "circle"
  ;set size particle-size
  set energy initial-gas-temperature
  set color-type 5
  set color color-type
  set mass (10)  ; atomic mass
  hatch 1 [
    set breed arrowheads
    set hidden? true
    create-link-from myself [ tie ]
  ]
  set speed speed-from-energy
  set last-collision nobody
end


; Place gas-particles at random, but they must not be placed on top of wall atoms.
; This procedure takes into account the fact that wall molecules could have two possible arrangements,
; i.e. high-surface area ot low-surface area.
to random-position ;; gas-particles procedure
  let open-patches nobody
  let open-patch nobody
  set open-patches patches with [not any? turtles-here and pxcor != max-pxcor and pxcor != min-pxcor and pycor != min-pycor and pycor != max-pycor]
  set open-patch one-of open-patches

  ; Reuven added the following "if" so that we can get through setup without a runtime error.
  if open-patch = nobody [
    user-message "No open patches found.  Exiting."
    stop
  ]

  setxy ([ pxcor ] of open-patch) ([ pycor ] of open-patch)
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; wall penetration error handling procedure
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; if gas-particles actually end up within the wall

to remove-from-walls
  let this-wall walls-here with [ not hidden? ]

  if count this-wall != 0 [
    let available-patches patches with [ not any? walls-here ]
    let closest-patch nobody
    if (any? available-patches) [
      set closest-patch min-one-of available-patches [ distance myself ]
      set heading towards closest-patch
      setxy ([ pxcor ] of closest-patch)  ([ pycor ] of closest-patch)
    ]
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;GRAPHS;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;REPORTERS;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to-report speed-from-energy
  report sqrt (2 * energy / mass)
end

to-report energy-from-speed
  report (mass * speed * speed / 2)
end

to-report limited-particle-energy
  let limited-energy energy
  if limited-energy > max-particle-energy [ set limited-energy max-particle-energy ]
  if limited-energy < min-particle-energy [ set limited-energy min-particle-energy ]
  report limited-energy
end

to-report vibration-displacement

end


to do-plots
  set-current-plot "Temperature of solids"
  if #-1-wall-particles > 0 [
    set-current-plot-pen "left"
    plotxy ticks avg-temp-type-1
  ]
    if #-2-wall-particles > 0 [
    set-current-plot-pen "middle"
    plotxy ticks avg-temp-type-2
  ]
    if #-3-wall-particles > 0 [
    set-current-plot-pen "right"
    plotxy ticks avg-temp-type-3
  ]
end



; Copyright 2006 Uri Wilensky.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
452
50
902
399
-1
-1
17.0
1
10
1
1
1
0
1
1
1
0
25
0
19
1
1
1
ticks
30.0

BUTTON
108
10
204
55
go/pause
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
4
10
104
55
setup/reset
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
552
11
733
44
middle.start.temp
middle.start.temp
-235
300
20.0
1
1
C
HORIZONTAL

PLOT
8
136
338
389
Temperature of solids
time
temperature (in C)
0.0
10.0
-250.0
350.0
true
true
"" ""
PENS
"left" 1.0 0 -6917194 true "" ""
"middle" 1.0 0 -955883 true "" ""
"right" 1.0 0 -13210332 true "" ""

SLIDER
392
11
550
44
left.start.temp
left.start.temp
-235
300
300.0
1
1
C
HORIZONTAL

SLIDER
735
11
907
44
right.start.temp
right.start.temp
-235
300
-235.0
1
1
C
HORIZONTAL

MONITOR
257
88
380
129
temp. change right
avg-temp-type-3 - init-temp#3
1
1
10

MONITOR
124
88
254
129
temp. change middle
avg-temp-type-2 - init-temp#2
1
1
10

MONITOR
6
88
120
129
temp. change left 
avg-temp-type-1 - init-temp#1
1
1
10

SLIDER
342
212
375
391
slide-left-solid-up
slide-left-solid-up
0
10
10.0
1
1
NIL
VERTICAL

SLIDER
416
212
449
390
slide-right-solid-up
slide-right-solid-up
0
10
10.0
1
1
NIL
VERTICAL

SLIDER
380
213
413
390
slide-middle-solid-up
slide-middle-solid-up
0
10
1.0
1
1
NIL
VERTICAL

MONITOR
213
10
270
51
time
ticks
1
1
10

@#$#@#$#@
## WHAT IS IT?

This model supports a drawing style interface for "sketching" up representations of new systems to explore related to gas behavior and gas particles.  This model is part of the "Connected Chemistry" curriculum http://ccl.northwestern.edu/curriculum/ConnectedChemistry/ which explores the behavior of gases.

Most of the models in the Connected Chemistry curriculum use the same basic rules for simulating the behavior of gases.  Each model highlights different features of how gas behavior is related to gas particle behavior.

In all of the models, gas particles are assumed to move and to collide, both with each other and with objects such as walls.

In this model, particles can be added, color coded, and sped up or slowed down, by drawing with the mouse cursor in the WORLD & VIEW.  Also, additional types of removable and replaceable walls can be added to the WORLD.

This model enables students to draw a model of a real world system and then test that model.  A wide range of real world systems can be modeled with this simple interface (e.g. diffusion of perfume from an uncapped container, hot gas mixed with a cold gas, mixtures of gases).

## HOW IT WORKS

The particles are modeled as hard balls with no internal energy except that which is due to their motion.  Collisions between particles are elastic.  Collisions with the wall are not.

The exact way two particles collide is as follows:
1. A particle moves in a straight line without changing its speed, unless it collides with another particle or bounces off the wall.
2. Two particles "collide" if they find themselves on the same patch. In this model, two turtles are aimed so that they will collide at the origin.
3. An angle of collision for the particles is chosen, as if they were two solid balls that hit, and this angle describes the direction of the line connecting their centers.
4. The particles exchange momentum and energy only along this line, conforming to the conservation of momentum and energy for elastic collisions.
5. Each particle is assigned its new speed, heading and energy.

As the walls of the box are heated, the sides of the walls will change color from a deep red (cool) to a bright red, to pink to a pale pink white (hot).  The walls contain a constant heat value throughout the simulation.

The exact way particles gain energy from the walls of the box is as follows:
1. Particles check their state of energy (kinetic).
2. They hit or bounce off the wall.
3. They find wall energy and set their new energy to be the average of their old kinetic energy and the wall energy.
4. They change their speed and direction after the wall hit.

## HOW TO USE IT

Buttons:
SETUP - sets up the initial conditions set on the sliders.
GO/STOP - runs and stops the model.
MOUSE INTERACTION - when this is set to "none - let the particles interact" the particles will move and interact with each other and the surroundings.  When set to any other value you can then click in the WORLD & VIEW to paint, erase, color, or add various objects and properties.

Sliders:
INITIAL-#-PARTICLES - sets the number of gas particles in the box when the simulation starts.
INITIAL-GAS-TEMPERATURE sets the initial temperature of the gas.

Switches:
SHOW-WALL-HITS? turn visualization of when particles hits the walls (as flashes) on or off

Choosers:
VISUALIZE-SPEED? allows you to visualize particle speeds.  For example, selecting "arrows", creates a representation of each particle velocity using a scalar arrow.  Selecting "shades" creates representation of each particle speed using a brighter (faster) or darker (slower) shade of the particle's color.

MOUSE-INTERACTION  sets the type interaction the user can do with the mouse in the WORLD & VIEW.  Possible settings include:
"none - let the particles interact" - particles move about
"draw basic wall" - adds a gray wall under the mouse cursor
"draw red removable wall" - adds a red wall under the mouse cursor which can be alternatively removed and replaced (like a valve) using the REMOVE/REPLACE RED WALL.
"draw green removable wall" - adds a green wall under the mouse cursor which can be alternatively removed and replaced (like a valve) using the REMOVE/REPLACE GREEN WALL.
"big eraser" - erases all objects (except the yellow box boundary walls) under the mouse cursor.
"slow down particles" - increase the current speed of the particles by 10%.
"speed up particles" - reduces the current speed of the particles by 10%.
"paint particles green" - recolors the particles under the mouse cursor green (other settings include orange and purple)
"add green particles" - adds a couple of new particles under the mouse cursor (other settings include orange and purple)

Plots:
- 1: TEMPERATURE OF GASES VS. TIME: plots the temperature of the different gases in the model, as indicated by their color (orange particles, green particles, and purple particles)

## THINGS TO NOTICE

The mouse interaction can be used while the model is running as well as when it is stopped.

## THINGS TO TRY

Create a model of how odors move throughout a room.  Why do some people smell the odor before others?  Does the layout of furniture, large objects, and walls in the room effect the movement of the odor?  How about the temperature of the air in the room?

Create a model of diffusion of a perfume from a closed container.  How would you represent the different gases (the perfume and the surrounding air)?  What shape will the container be?  How will you model a removable cap or lid?

Create a model of room filled with cold air and a different room filled with warm air.  How will represent these different rooms of air?  What could you add to show what happens when they mix?

Create a model of heat transfer that shows what happens to the energy of one very fast moving gas particle when it hits a bunch of very slow moving gas particles.  What does this show happening to the energy of the initial gas particles?

## RELATED MODELS

See GasLab Models
See other Connected Chemistry models.

## CREDITS AND REFERENCES

This model is part of the Connected Chemistry curriculum.  See http://ccl.northwestern.edu/curriculum/chemistry/.

We would like to thank Sharona Levy and Michael Novak for their substantial contributions to this model.

## HOW TO CITE

If you mention this model or the NetLogo software in a publication, we ask that you include the citations below.

For the model itself:

* Wilensky, U. (2006).  NetLogo Connected Chemistry 8 Gas Particle Sandbox model.  http://ccl.northwestern.edu/netlogo/models/ConnectedChemistry8GasParticleSandbox.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Please cite the NetLogo software as:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

To cite the Connected Chemistry curriculum as a whole, please use:

* Wilensky, U., Levy, S. T., & Novak, M. (2004). Connected Chemistry curriculum. http://ccl.northwestern.edu/curriculum/chemistry/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2006 Uri Wilensky.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.

<!-- 2006 ConChem -->
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

carbon
true
0
Circle -1184463 true false 68 83 134

carbon-activated
true
0
Circle -1184463 true false 68 83 134
Line -2674135 false 135 90 135 210

carbon2
true
0
Circle -955883 true false 30 45 210

circle
false
1
Circle -2674135 true true 30 30 240

circle 2
false
0
Circle -7500403 true true 16 16 270
Circle -16777216 true false 46 46 210

clock
true
0
Circle -7500403 true true 30 30 240
Polygon -16777216 true false 150 31 128 75 143 75 143 150 158 150 158 75 173 75
Circle -16777216 true false 135 135 30

co2
true
0
Circle -13791810 true false 83 165 134
Circle -13791810 true false 83 0 134
Circle -1184463 true false 83 83 134

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

eraser
false
0
Rectangle -7500403 true true 0 0 300 300

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

heater-a
false
0
Rectangle -7500403 true true 0 0 300 300
Rectangle -16777216 true false 90 90 210 210

heater-b
false
0
Rectangle -7500403 true true 0 0 300 300
Rectangle -16777216 true false 30 30 135 135
Rectangle -16777216 true false 165 165 270 270

hex
false
0
Polygon -7500403 true true 0 150 75 30 225 30 300 150 225 270 75 270

hex-valve
false
0
Rectangle -7500403 false true 0 0 300 300
Polygon -7500403 false true 105 60 45 150 105 240 195 240 255 150 195 60

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

new-line
false
0
Line -7500403 true 150 0 150 15
Line -7500403 true 150 60 150 75
Line -7500403 true 150 30 150 45
Line -7500403 true 150 90 150 105
Line -7500403 true 150 120 150 135
Line -7500403 true 150 150 150 165
Line -7500403 true 150 180 150 195
Line -7500403 true 150 210 150 225
Line -7500403 true 150 240 150 255
Line -7500403 true 150 270 150 285

nitrogen
true
0
Circle -10899396 true false 83 135 134
Circle -10899396 true false 83 45 134

oxygen
true
0
Circle -13791810 true false 83 135 134
Circle -13791810 true false 83 45 134

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

rounded-hexagon
true
0
Circle -7500403 true true 89 3 120
Circle -7500403 true true 15 135 120
Circle -7500403 true true 165 135 120
Circle -7500403 true true 15 45 120
Circle -7500403 true true 165 45 120
Circle -7500403 true true 89 175 120
Polygon -7500403 true true 14 99 15 200 47 248 116 285 179 287 258 246 286 190 286 104 254 51 182 10 117 9 47 49

rounded-pentagon
false
0
Polygon -2674135 true false 30 107 127 30 172 30 270 100 286 145 251 266 217 291 80 290 50 266 20 143
Circle -2674135 true false 119 21 64
Circle -2674135 true false 19 102 58
Circle -2674135 true false 218 97 70
Circle -2674135 true false 186 224 66
Circle -2674135 true false 49 223 66

rounded-square
true
0
Circle -7500403 true true 14 14 212
Circle -7500403 true true 74 14 212
Circle -7500403 true true 74 74 212
Circle -7500403 true true 14 74 212
Polygon -7500403 true true 14 187 14 111 113 13 184 13 287 114 287 183 185 287 112 286

rounded-triangle
true
0
Circle -7500403 true true 89 7 120
Circle -7500403 true true 15 135 120
Circle -7500403 true true 165 135 120
Polygon -7500403 true true 100 30 23 165 85 255 220 255 276 162 200 33

spray paint
false
0
Rectangle -7500403 false true 0 0 300 300
Circle -7500403 false true 75 75 150

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

square2
true
0
Polygon -7500403 true true 150 0 75 0 30 30 0 75 0 150 0 225 30 270 75 300 225 300 270 270 300 225 300 75 270 30 225 0

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

valve-1
false
0
Rectangle -7500403 false true 0 0 300 300
Rectangle -7500403 false true 120 120 180 180

valve-2
false
0
Rectangle -7500403 false true 0 0 300 300
Rectangle -7500403 false true 60 120 120 180
Rectangle -7500403 false true 165 120 225 180

valve-hex
false
0
Rectangle -7500403 false true 0 0 300 300
Polygon -7500403 false true 105 60 45 150 105 240 195 240 255 150 195 60

valve-triangle
false
0
Rectangle -7500403 true true 0 0 300 300
Polygon -16777216 true false 150 45 30 240 270 240

valves
false
0
Rectangle -7500403 false true 0 0 300 300

wall
false
0
Rectangle -7500403 true true 0 0 300 300

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
