extensions [ sound ]

globals [
 chromosomes ;; list of all possible chromosomes
 random-whos ;; keeps a shuffled list for pretty-ness
 generations ;; number of generations we've seen
 available-drummers ;; list of currently-available drummers
 fake-who ;; global variable to fix hubnet bug
]

turtles-own [
  my-who ;; variable to fix hubnet
  my-fitness ;; fitness of a drummer
  my-chromosomes my-pattern ;; lists to hold the chromosomes and hit-pattern of a drummer
  my-velocity my-instrument ;; the (int) velocity value and (string) MIDI instrument for that turtle
  mutation-rate ;; variables to control "reproduction"
  hits ;; counts the number of drum hits for a turtle
  hits-since-evolve ;; the number of hits since a mutation or evolution
]

breed [ students student ] ;; for hubnet
students-own [
 my-drummer-who ;; the turtle player that is associated with this student
 user-id ;; students choose a user name so we keep that here
]
breed [ low-drums low-drummer ]
breed [ med-drums med-drummer ]
breed [ high-drums high-drummer ]

;; Hubnet Code
to startup
  hubnet-reset
end

;; when a new user logs in create a student turtle
;; this turtle will store any state on the client
;; values of sliders, etc.
to create-new-student
  create-students 1
  [
    ;; store the message-source in user-id now
    ;; so when you get messages from this client
    ;; later you will know which turtle it affects
    set user-id hubnet-message-source
    set label user-id
    ;; initialize turtle variables to the default
    ;; value of the corresponding widget in the client interface


    set my-drummer-who one-of available-drummers

    set available-drummers remove my-drummer-who available-drummers
    ;; SET DEFAULT INSTRUMENT

    if is-low-drummer? one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [
       ask one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [ set my-instrument "BASS DRUM 1" ]
    ]

    if is-med-drummer? one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [
       ask one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [ set my-instrument "LOW CONGA" ]
    ]

    if is-high-drummer? one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [
       ask one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [ set my-instrument "CLOSED HI HAT" ]
    ]

    ;; update the clients with any information you have set
    send-info-to-clients
  ]
end


to listen-clients
  ;; KICK ANYONE THAT IS OVER THE 16

  ;; as long as there are more messages from the clients
  ;; keep processing them.
  while [ hubnet-message-waiting? ]
  [
    ;; get the first message in the queue
    hubnet-fetch-message
    ifelse hubnet-enter-message? and not empty? available-drummers  ;; when clients enter we get a special message
    [ create-new-student ]
    [
      ifelse hubnet-exit-message? ;; when clients exit we get a special message
      [ remove-student ] ;; need to add this drummer back to the available list
      [ ask students with [user-id = hubnet-message-source]
        [ execute-command hubnet-message-tag ] ;; otherwise the message means that the user has
      ]                                        ;; done something in the interface hubnet-message-tag                                         ;; is the name of the widget that was changed
    ]
  ]
end


;; Other messages correspond to users manipulating the
;; client interface, handle these individually.
to execute-command [command] ;; turtle procedure
  ;; you should have one if statement for each widget that
  ;; can affect the outcome of the model, buttons, sliders, switches
  ;; choosers and the view, if the user clicks on the view you will receive
  ;; a message with the tag "View" and the hubnet-message will be a
  ;; two item list of the coordinates
  if command = "my-instrument"
  [
    ;; note that the hubnet-message will vary depending on
    ;; the type of widget that corresponds to the tag
    ;; for example if the widget is a slider the message
    ;; will be a number, of the widget is switch the message
    ;; will be a boolean value
    ask one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [set my-instrument hubnet-message]
    stop
  ]
  if command = "my-fitness" [ ask one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [ set my-fitness hubnet-message stop ] ]
end

;; whenever something in world changes that should be displayed in
;; a monitor on the client send the information back to the client
to send-info-to-clients ;; turtle procedure
  hubnet-send user-id "hits-since-evolved" [hits-since-evolve] of one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself]
  hubnet-send user-id "my-instrument" [my-instrument] of one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself]

  let temp ""

  if is-low-drummer? one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [ set temp "low" ]
  if is-med-drummer? one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [ set temp "med" ]
  if is-high-drummer? one-of turtles with [(not is-student? self) and who = [my-drummer-who] of myself] [ set temp "high" ]

  hubnet-send user-id "my-part" temp
end

;; when a user logs out make sure to clean up the turtle
;; that was associated with that user (so you don't try to
;; send messages to it after it is gone) also if any other
;; turtles of variables reference this turtle make sure to clean
;; up those references too.
to remove-student

  ask students with [user-id = hubnet-message-source] [
     set available-drummers fput my-drummer-who available-drummers
     die
  ]
end

to setup
  hubnet-kick-all-clients
  ca
  cp
  cd
  clear-output
  ;; Make the view big enough to show 16 lines and however many 'beats'
  resize-world 0 ((num-chromosomes * 4) - 1) 0 15
  set-globals
  set-initial-turtle-variables
  reset-ticks
  update-view
end

;; Method to play a pattern without any evolution
to go-no-evolve
  listen-clients
  ask turtles with [not is-student? self] [
    play ;; includes end of life
  ]
  update-view
  ask students [send-info-to-clients]
  tick
  wait 60 / TEMPO-BPM / 4
end

;; Method to play a pattern with evolution
to go
  listen-clients

  ;; If we've reached the end of a pattern, do some evolution!
  if (ticks mod ((num-chromosomes * 4)) = 0) and (ticks != 0) [
    set generations generations + 1
    go-evolve
  ]

  foreach available-drummers [
   ask one-of turtles with [my-who = ?] [ set my-fitness random 100 ]
  ]

  ask turtles with [not is-student? self] [
   play
  ]
  update-view
  ask students [send-info-to-clients]
  tick
  wait 60 / TEMPO-BPM / 4   ;; This roughly sets temp
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; PLAY FUNCTIONS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to play-my-drum ;; turtle proceudre
  let temp my-velocity
  if sound? [
    if solo? [ ;; If you're a soloer, play out! Otherwise, SHHHH.
      ifelse my-who = soloer [
        set temp my-velocity + 50
      ]
      [
        set temp my-velocity - 50
      ]
   ]
    sound:play-drum my-instrument temp
  ]
end

to play ;; turtle procedure
  if is-low-drummer? self [
    if item (ticks mod (num-chromosomes * 4)) my-pattern = 1 [
      play-my-drum
      set hits hits + 1
      set hits-since-evolve hits-since-evolve + 1
    ]
  ]

  if is-med-drummer? self [
    if item (ticks mod (num-chromosomes * 4)) my-pattern = 1 [
      play-my-drum
      set hits hits + 1
      set hits-since-evolve hits-since-evolve + 1
    ]
  ]

  if is-high-drummer? self [
    if item (ticks mod (num-chromosomes * 4)) my-pattern = 1 [
      play-my-drum
      set hits hits + 1
      set hits-since-evolve hits-since-evolve + 1
    ]
  ]
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; END PLAY FUNCTIONS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; EVOLUTION FUNCTIONS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go-evolve
  ;; If there isn't a soloist, ask 2 of each type to evolve
  ifelse not solo? [
    ask n-of 2 low-drums [
      evolve
    ]
    ask n-of 2 med-drums [
      evolve
    ]
    ask n-of 2 high-drums [
      evolve
    ]
    ;; If a drummer hasn't changed in a while, mutate
    ask turtles with [(not is-student? self) and hits-since-evolve > hit-limit] [
      mutate
      set hits-since-evolve 0
    ]
  ] [ ;; If there is a soloist, do the same, but don't include the soloer
   ask n-of 2 low-drums with [my-who != soloer] [
      evolve
    ]
    ask n-of 2 med-drums with [my-who != soloer] [
      evolve
    ]
    ask n-of 2 high-drums with [my-who != soloer]  [
      evolve
    ]
    ;; If a drummer hasn't changed in a while, mutate
    ask turtles with [(not is-student? self) and hits-since-evolve > hit-limit and my-who != soloer] [
      mutate
      set hits-since-evolve 0
    ]
  ]
end

to evolve ;; turtle procedure
 let mate nobody
 let list-of-fitnesses []
 let search-fitness 0
 if is-low-drummer? self [
   set list-of-fitnesses [fitness] of other breed
   set search-fitness select-random-weighted-fitness list-of-fitnesses
   set mate one-of other breed with [fitness = search-fitness]
 ]

 if is-med-drummer? self [
   set list-of-fitnesses [fitness] of turtles with [(not is-student? self) and breed != [breed] of myself]
   set search-fitness select-random-weighted-fitness list-of-fitnesses
   set mate one-of turtles with [(breed != [breed] of myself) and (fitness = search-fitness)]
 ]

 if is-high-drummer? self [
   set list-of-fitnesses [fitness] of other breed
   set search-fitness select-random-weighted-fitness list-of-fitnesses
   set mate one-of other breed with [fitness = search-fitness]
 ]

 let offspring-chromosomes reproduce-with mate

 ask min-one-of other breed with [(not is-student? self) and my-who != soloer] [fitness] [
   set my-chromosomes offspring-chromosomes
   update-pattern
   set hits-since-evolve 0
 ]
end

;; This is where the basic genetic algorithm comes in
to-report reproduce-with [mate] ;; turtle procedure
  ;; ASKER IS 1st Parent MATE is 2nd parent
  ;;; my-chromosomes
  let her-chromosomes [my-chromosomes] of mate

  ;; Pick a random cross-over point
  let crossover-point random length my-chromosomes

  ;; Combine the chromosomes
  let baby-chromosomes sentence (sublist my-chromosomes 0 crossover-point) (sublist her-chromosomes crossover-point length her-chromosomes)

  ;; Do a little mutation
  let mutation-chance 0
  if is-low-drummer? self [
    set mutation-chance 50
  ]
  if is-med-drummer? self [
    set mutation-chance 25
  ]
  if is-high-drummer? self [
    set mutation-chance 10
  ]

  ;; Maybe actually mutate
  if random 100 > mutation-chance [
    set baby-chromosomes mutate-chromosomes baby-chromosomes
  ]
  report baby-chromosomes
end


;; FITNESS FUNCTIONS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Dependent on breed, because you can lose fitness or gain fitness by fitting to your particular proclivities
to-report fitness ;; turtle procedure
  ;; Arbirtrary 10% window around target density
 ;; use add 1 smoothing
 report my-fitness + 1
end

to mutate ;; turtle procedure
  set my-chromosomes mutate-chromosomes my-chromosomes
  update-pattern
end

;; Method to mutate a chromosome
to-report mutate-chromosomes [the-chromosomes]
  ;; basically picks a chromosome, mutates it, returns a new set
  let new-chromosomes the-chromosomes
  repeat num-mutations [
    let temp random num-chromosomes
    set new-chromosomes replace-item temp new-chromosomes ((round (random-normal (item temp new-chromosomes) mutation-strength)) mod 16)
  ]
  report new-chromosomes
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; END EVOLUTION FUNCTIONS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; HELPER FUNCTIONS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Called after a chromosome update in order to redefine that turtle's pattern
to update-pattern ;; turtle method
  set my-pattern []
  foreach my-chromosomes [
   set my-pattern sentence my-pattern (get-chromosome ?)
  ]
end

to set-globals
  set fake-who 0
  set available-drummers n-values 16 [?]
  set generations 0
  set random-whos n-values 16 [?]
  ;; this is just for looks.
  if shuffle-parts? [
    set random-whos shuffle random-whos
    set random-whos shuffle random-whos
  ]
  set chromosomes []
  ;; CHROMOSOME LIBRARY
  let c0 [0 0 0 0]
  let c1 [1 0 0 0]  let c2  [0 1 0 0] let c3 [0 0 1 0] let c4 [0 0 0 1]
  let c5 [1 1 0 0]  let c6  [1 0 1 0] let c7 [1 0 0 1] let c8 [0 1 1 0]
  let c9 [0 1 0 1]  let c10 [0 0 1 1]
  let c11 [1 1 1 0] let c12 [1 0 1 1] let c13 [1 1 0 1] let c14 [0 1 1 1]
  let c15 [1 1 1 1]
  set chromosomes (list c0 c1 c2 c3 c4 c5 c6 c7 c8 c9 c10 c11 c12 c13 c14 c15)
  ;; END CHROMOSOME LIBRARY
end

to set-initial-turtle-variables
  create-low-drums 6 [
    set my-who fake-who
    set fake-who fake-who + 1
    set my-instrument "LOW CONGA"
    set my-velocity 85
    set color red
    set mutation-rate 64
  ]

  create-med-drums 5 [
    set my-who fake-who
    set fake-who fake-who + 1
    set my-instrument "Bass Drum 1"
    set color green
    set my-velocity 85
    set mutation-rate 32
  ]

  create-high-drums 5 [
    set my-who fake-who
    set fake-who fake-who + 1
    set my-instrument "Closed Hi Hat"
    set color blue
    set my-velocity 85
    set mutation-rate 16
  ]

  ask turtles with [(not is-student? self)] [
    set my-pattern []
    set my-chromosomes (n-values num-chromosomes [1])
    ht
    update-pattern
    set hits 0
  ]
end

;; Method to update the view (simplified music notation)
to update-view
 ask turtles with [not is-student? self] [
   let temp 0
   let row item my-who random-whos
   foreach my-pattern [
     ifelse ? = 1 [
       ifelse solo? and (soloer = my-who) [
         ask patch temp row [set pcolor white]
       ] [
         ask patch temp row [set pcolor [color] of myself]
       ]
     ] [
     ask patch temp row [set pcolor black]
     ]
     set temp temp + 1
   ]
 ]
 ask patches with [pxcor = (ticks mod (num-chromosomes * 4))] [set pcolor yellow]

 ask students [
   let row item my-drummer-who random-whos
   hubnet-send-override user-id patches with [pycor != row] "pcolor" [ black ]
 ]
end

;; Method to get a chromosomes pattern from the library
to-report get-chromosome [index]
  report item index chromosomes
end

;; This is my version picking a weighted random turtle
to-report select-random-weighted-fitness [theList]
   let weighted-list []
   foreach theList [
     ;; add one smoothing
     let temp ?
     foreach n-values round ((? / sum theList * 100) + 1) [?] [
       set weighted-list fput temp weighted-list
     ]
   ]
   report item (random length weighted-list) weighted-list
end

;;; OLD CODE
;breed [ low-drums low-drummer ]
;;; Atsimevu
;;; 64 Low Conga
;;; 63 Open Hi Conga
;;; 52 Mute Hi Conga
;;;  open-hand open-fingers slap-fingers slap-two-fingers ;; Options for hit
;
;;; Sogo or Kidi
;;; MIDI INSTRUMENT 117. Taiko Drum
;breed [ med-drums med-drummer ]
;
;;; Kagan
;;; 65. Hi Timbale
;breed [ high-drums high-drummer ]
@#$#@#$#@
GRAPHICS-WINDOW
290
10
780
361
-1
-1
20.0
1
10
1
1
1
0
0
0
1
0
23
0
15
1
1
1
ticks
30.0

BUTTON
5
50
71
83
setup
setup\nsound:play-note \"TRUMPET\" 60 62 2\nwait 2
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
5
10
150
43
num-chromosomes
num-chromosomes
2
12
6
1
1
NIL
HORIZONTAL

SWITCH
155
10
285
43
shuffle-parts?
shuffle-parts?
1
1
-1000

PLOT
545
370
780
520
Hits per Drummer
NIL
NIL
0.0
16.0
0.0
10.0
true
false
"" ""
PENS
"low" 1.0 1 -2674135 true "" "plot-pen-reset\nask low-drums [ plotxy who hits ]"
"med" 1.0 1 -10899396 true "" "plot-pen-reset\nask med-drums [plotxy who hits]"
"high" 1.0 1 -13345367 true "" "plot-pen-reset\nask high-drums [plotxy who hits]"

PLOT
5
370
285
520
Average Fitness
NIL
NIL
0.0
1.0
0.0
1.0
true
true
"" ""
PENS
"all" 1.0 0 -16777216 true "" "plot mean [fitness] of turtles"
"low" 1.0 0 -2674135 true "" "plot mean [fitness] of low-drums"
"med" 1.0 0 -10899396 true "" "plot mean [fitness] of med-drums"
"high" 1.0 0 -13345367 true "" "plot mean [fitness] of high-drums"

SWITCH
5
90
108
123
sound?
sound?
0
1
-1000

SLIDER
5
210
130
243
num-mutations
num-mutations
1
10
2
1
1
NIL
HORIZONTAL

SLIDER
135
210
285
243
mutation-strength
mutation-strength
0
8
1
1
1
NIL
HORIZONTAL

CHOOSER
40
310
132
355
soloer
soloer
0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15
9

MONITOR
40
255
130
300
NIL
generations
17
1
11

BUTTON
75
50
210
83
go-once-no-evolve
repeat num-chromosomes * 4 [go-no-evolve]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
215
50
285
83
go-once
repeat num-chromosomes * 4 [ go ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
135
315
238
348
solo?
solo?
1
1
-1000

SLIDER
5
130
285
163
tempo-bpm
tempo-bpm
40
200
200
1
1
NIL
HORIZONTAL

MONITOR
135
255
222
300
Density
precision (count patches with [(pcolor != black) and (pcolor != yellow)] / (count patches)) 3
17
1
11

PLOT
290
370
540
520
Hits Since Evolution
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"low" 1.0 1 -2674135 true "" "plot-pen-reset \nask low-drums [ plotxy who hits-since-evolve ]"
"pen-1" 1.0 1 -10899396 true "" "plot-pen-reset ask med-drums [ plotxy who hits-since-evolve ]"
"pen-2" 1.0 1 -13345367 true "" "plot-pen-reset ask high-drums [ plotxy who hits-since-evolve ]"

SLIDER
5
170
130
203
hit-limit
hit-limit
50
200
200
1
1
NIL
HORIZONTAL

SLIDER
135
170
285
203
hit-density
hit-density
10
90
50
1
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This model aims to demonstrate the concept of a genetic algorithm in the context of rhythm production. Unlike the other GenJam models, this model doesn't rely on fitness functions inspired by ethnomusicology research. instead, we still have low, medium, and high-drums, but each user gets to decide the fitness of their pattern in order to work in concert with evolution to create jamming beats. This is a 'duple' version, where there are 4 'beats' per chromosome.

## HOW IT WORKS

There are 16 drums, 5 high drummers (INSTRUMENT), 5 medium drummers (INSTRUMENT), and 6 low drummers (INSTRUMENT). Note that we use "low", "medium", and "high" to describe the pitch and tembre of the drum. Each drummer has a set of "rhythm chromosomes" which dictate what pattern it plays.

This model works exactly like GenJam-Duple (Triple and Mixed) except that fitness is determined by each of the Hubnet Clients!

## HOW TO USE IT

The whole idea of this model is to experiement with the parameters in order to build a "great" rhythm. What makes a great rhythm is entirely up to you.

There are two interface elements that must be set before pressing the SETUP button:

1. NUM-CHROMOSOMES This specifies the number of rhythmic chromosomes each player has
2. SHUFFLE-PARTS? This is simply a GUI change that shuffles around the player's lines on the view (just to make things look cooler)

GO-ONCE is used to ask all the turtles to play their pattern exactly once and then evolve
GO-ONCE-NO-EVOLVE can be used to play any particular pattern exactly once with no evolution

SOUND? is used to toggle sound output

The following are options that the learner can tweak to modify the evolution process:

TEMPO-BPM changes how fast each pattern is played (measured in beats (or chromosomes) per minute)
HIT-LIMIT defines how long a turtle can go without being forced to evolve in someway
(Note: again, this is a deviation from the traditional genetic algorithm but can be used to escape 'stale' rhythms)
HIT-DENSITY-MODIFIER is a modifier to specify how "dense" a pattern should be (hits vs rests)
NUM-MUTATIONS is how many mutations are applied to an off-springs chromosomes
MUTATION-STRENGTH dictates how much a particular chromosome can mutate
SOLOER is a chooser that allows you to single out a player
SOLO? dictates whether or not to allow SOLOER to solo

The following are outputs of the model:

GENERATIONS counts how many generations of players we've seen
DENSITY represents the current ratio of hits to rests

The following are plots of the model:

AVERAGE FITNESS plots the average fitness of each type of turtle and the average overall fitness over time
HITS SINCE EVOLUTION is a histogram showing how long it has been since each player has 'evolved'
HITS PER DRUMMER is a histogram showing how many hits each player has done

## THINGS TO NOTICE

When you first start the model, everyone starts with the same chromosomes. So if you disable mutation by making MUTATION-STRENGTH 0, your model won't evolve! That's because the same chromosomes are just being combined together for each player.

Notice that sometimes, a single turtle sticks around for a really long time (because it's super fit!)

Checkout the Hits per Drummer graph. Do you see any general trends over the course of running the model? Why might that trend be taking place?

Notice that the SOLO function doesn't just make one player louder, it makes it so that player doesn't evolve or mutate. How does that affect the model?

## THINGS TO TRY

Use the SOLO function to force a rhythm to evolve around a single stagnant player.

Try and make a rhythm where all three types of players are "equally" fit!

Try to evolve a rhythm where the high drums are very sparse.

See how creating a "complex" meter changes the way you hear the beat. An easy way to do this is to set the NUMBER-CHROMOSOMES to 5 and see if you can hear the musical phrase.

## EXTENDING THE MODEL

HubNet Models are pretty complicated, but see if you can make it so n clients can connect!

## NETLOGO FEATURES

Notice that this model purposefully slows down the NetLogo world. That's because music doesn't happen as fast as possible! Unfortunately, this doesn't work for the first tick, so the space between the "first beat" and "second beat" is not equal to the space betwen the rest of the beats.

In addition, we do some HubNet magic to only show each person's pattern on their HubNet Client view.

## RELATED MODELS

To play with other rhythmic feels, check out GenJam-Duple, GenJam-Triple and GenJam-Mixed. Also, check out the Sound Machines and Drum Machine models!

## CREDITS AND REFERENCES

Author: Connor Bain
Written for Multi-agent Modeling 2016.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 6.0-M6
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
VIEW
215
10
775
330
0
0
0
1
1
1
1
1
0
1
1
1
0
23
0
15

CHOOSER
10
65
205
110
my-instrument
my-instrument
"ACOUSTIC BASS DRUM" "CLOSED HI HAT" "OPEN HI HAT" "BASS DRUM 1" "ACOUSTIC SNARE" "LOW FLOOR TOM" "HAND CLAP" "LOW TOM" "HI MID TOM" "RIDE CYMBAL 1" "CHINESE CYMBAL" "RIDE BELL" "TAMBOURINE" "SPLASH CYMBAL" "COWBELL" "VIBRASLAP" "HI BONGO" "LOW BONGO" "MUTE HI CONGA" "OPEN HI CONGA" "LOW CONGA" "HI AGOGO" "LOW AGOGO" "CABASA" "MARACAS" "CLAVES" "SHORT GUIRO" "SHORT WHISTLE" "MUTE CUICA" "MUTE TRIANGLE"
0

MONITOR
10
10
77
59
my-part
NIL
3
1

SLIDER
12
115
202
148
my-fitness
my-fitness
0
100
49
1
1
NIL
HORIZONTAL

MONITOR
80
10
207
59
hits-since-evolved
NIL
3
1

@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
1
@#$#@#$#@
