;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

breed [particles particle]            ;;the particles exist until deposition.
breed [active-sites active-site]      ;;these sites are the active zone's

active-sites-own [height]             ;;holds the height of the active zone

globals [
  max-height  
  avg-height                          ;;the 4 of them regard the active zone
  width 
  step-size
  
  number-of-deposited                
  number-of-pores
]
 
;;bottom left corner of the world is the origin (0,0)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 
to setup    
  ca
  
  ;;initially I apply an even coat on the surface
  
  set step-size n-values (max-pxcor + 1) [0]  
  set avg-height 2
  set width 0
  set number-of-deposited max-pycor + 1
  set number-of-pores 0
    
  create-active-sites max-pxcor + 1 [
    if (hidden-zone = true) [hide-turtle]  
    set shape "circle"
    set size 1.0
    set color red
    set height 2
    setxy who 1]

  ask patches with [pycor = 0] [
    set pcolor green]
    
  reset-ticks
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go   
  
  set max-height max [height] of active-sites

  ;;to end the simulation when the ceiling is reached    
  if  max-height = max-pycor + 2 [       
    stop]  
  
  make-new-particle
  
  ;;it considers earlier the particles with lower height, else 
  ;;they don't stick properly
  foreach sort particles [ ask ? [      
    let psi 0
    let chi 0
    
    ifelse (next-nearest = false)
    [ 
    ;;first checks if a particle sticks immediately upon creation
    if any? neighbors4 with [pcolor = green] [
      set pcolor green
      set psi ycor
      set chi xcor 
      
      ;;either the particle is not the last one to be deposited or... 
      ifelse (psi < max-pycor)
      [ ask active-sites with [xcor = chi] [
          setxy chi psi + 1       ;;raise the height of the zone at the column
          set height psi + 2
        ]        
        ask active-sites with [xcor = chi - 1] [
          if (ycor < psi) [
            setxy xcor psi      ;;check if the left neighbor's is raised
            set height psi + 1  
          ]
        ]        
        ask active-sites with [xcor = chi + 1] [
          if (ycor < psi) [
            setxy xcor psi      ;;check if the right neighbor's is raised
            set height psi + 1  
          ]
        ]
        ;;this code block helps to calculate the density = (green patches)/((green patches)+(gray-4.9 patches))
        ask patches with [(pycor < psi) and (pxcor = chi) and (pcolor = black)] [ 
          set pcolor gray - 4.9]
      ] 
      ;;it is the last particle before the end of the simulation 
      [ ask active-sites with [xcor = chi] [
          hide-turtle
          set height psi + 2
        ]        
        ask active-sites with [xcor = chi - 1] [
          if (ycor < psi) [
            setxy xcor psi
            set height psi + 1  
          ]
        ]        
        ask active-sites with [xcor = chi + 1] [
          if (ycor < psi) [
            setxy xcor psi
            set height psi + 1  
          ]
        ]
        ask patches with [(pycor < psi) and (pxcor = chi) and (pcolor = black)] [
          set pcolor gray - 4.9]
      ]
      die                       ;;in the end, a stuck particle is killed, been                                
      ]                         ;;"transformed" to a green patch       
    ]   
    [
      if any? neighbors with [pcolor = green] [
      set pcolor green
      set psi ycor
      set chi xcor 
      
      ifelse (psi < max-pycor)
      [ ask active-sites with [xcor = chi] [
        setxy chi psi + 1       
        set height psi + 2
        ]        
        ask active-sites with [xcor = chi - 1] [
          setxy xcor psi + 1   
          set height psi + 2  
        ]        
        ask active-sites with [xcor = chi + 1] [
          setxy xcor psi + 1     
          set height psi + 2  
        ]
        ask patches with [(pycor < psi) and (pxcor = chi) and (pcolor = black)] [
          set pcolor gray - 4.9]
      ] 
      [ ask active-sites with [xcor = chi] [
          hide-turtle
          set height psi + 2
        ]        
        ask active-sites with [xcor = chi - 1] [
          hide-turtle
          set height psi + 2  
        ]        
        ask active-sites with [xcor = chi + 1] [
          hide-turtle
          set height psi + 2  
        ]
        ask patches with [(pycor < psi) and (pxcor = chi) and (pcolor = black)] [
          set pcolor gray - 4.9]
      ]
      die                                              
      ] 
    ]                      
    
    ;;if it is not the case of sticking immediately, you let it fall
    fall
    
    ;;and check if it sticks
    ifelse (next-nearest = false) 
    [  if any? neighbors4 with [pcolor = green] [
        set pcolor green
        set psi ycor
        set chi xcor 
        ask active-sites with [xcor = chi] [
          setxy chi psi + 1
          set height psi + 2
        ]
        ask active-sites with [xcor = chi - 1] [
          if (ycor < psi) [
            setxy xcor psi
            set height psi + 1  
          ]
        ]
        ask active-sites with [xcor = chi + 1] [
          if (ycor < psi) [
            setxy xcor psi
            set height psi + 1  
          ]
        ] 
        ask patches with [(pycor <= psi) and (pxcor = chi) and (pcolor = black)] [
          set pcolor gray - 4.9]     
        die
       ]
   ]
   [  if any? neighbors with [pcolor = green] [
        set pcolor green
        set psi ycor
        set chi xcor 
        ask active-sites with [xcor = chi] [
          setxy chi psi + 1
          set height psi + 2
        ]
        ask active-sites with [xcor = chi - 1] [
          setxy xcor psi + 1
          set height psi + 2 
        ]
        ask active-sites with [xcor = chi + 1] [
          setxy xcor psi + 1
          set height psi + 2 
        ]  
        ask patches with [(pycor < psi) and (pxcor = chi) and (pcolor = black)] [
          set pcolor gray - 4.9]   
        die
      ]
   ]  
  ]]
         
  tick
  update-variables
  do-plots  
end



to make-new-particle 
  create-particles 1 [
    set size 1.5
    set shape "square"
    set color green
    ifelse (shoot-near = false) [          
      setxy random-pxcor max-pycor]        ;;you can shoot particles from the ceiling, or
      [setxy random-pxcor max-height - 1]  ;;from the heighest of the active zone.
    set heading 180                        ;;particles head down
  ]  
end



to fall
  fd 1
end



to update-variables
   set width standard-deviation [height] of active-sites
   set avg-height mean [height] of active-sites 
   set number-of-deposited (count patches with [pcolor = green])  
   set number-of-pores (count patches with [pcolor = (gray - 4.9)])
   
   ;;for the distribution of step sizes...
   set step-size []               
   let i 0
   while [i < (max-pxcor + 1)] [          ;;for each of the columns...
    let next (i + 1)
    if (next = max-pxcor + 1) [           
      set next 0]                         ;;periodic boundary conditions...
    let diff (([height] of active-site i) - ([height] of active-site next))  
    set step-size (lput abs diff step-size)   ;;the difference in height is "diff"
    set i (i + 1)
   ]
end



to do-plots
  if (ticks mod 10 = 1)
  [
    set-current-plot "Avg. Height of Active Zone"
    set-current-plot-pen "Sur"
    plotxy number-of-deposited avg-height     
     
    set-current-plot "Width of Active Zone"
    set-current-plot-pen "Sur"
    plotxy number-of-deposited width
  ]
  ;; the histogram's code is at its "Edit..." window
end
@#$#@#$#@
GRAPHICS-WINDOW
606
10
956
381
-1
-1
2.27
1
10
1
1
1
0
1
0
1
0
149
0
149
1
1
1
ticks
30.0

BUTTON
7
10
82
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
87
10
162
43
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
6
65
291
215
Avg. height of Active Zone
Particles deposited
Avg. Height
1.0
10.0
1.0
10.0
true
false
"" ""
PENS
"Sur" 1.0 0 -16777216 true "" ""

PLOT
310
65
598
215
Width of Active Zone
Particles deposited
Width
1.0
10.0
0.0
10.0
true
false
"" ""
PENS
"Sur" 1.0 0 -16777216 true "" ""

SWITCH
169
10
305
43
shoot-near
shoot-near
1
1
-1000

SWITCH
316
10
451
43
hidden-zone
hidden-zone
0
1
-1000

PLOT
163
227
450
379
Step sizes of Active Zone
Step size
Frequency
0.0
10.0
0.0
10.0
true
false
"set-plot-x-range 0 10\nset-plot-y-range 0 10\nset-histogram-num-bars 10" ""
PENS
"default" 1.0 1 -16777216 true "set-plot-x-range 0 10\nset-plot-y-range 0 10\nset-histogram-num-bars 10" "histogram step-size"

SWITCH
460
10
596
43
next-nearest
next-nearest
1
1
-1000

MONITOR
464
227
598
272
Aggregate density
number-of-deposited / (number-of-deposited + number-of-pores)
3
1
11

@#$#@#$#@
## WHAT IS IT?


The morphological characteristics of a material's surface affect its properties. One of the simplest models to study the surface geometry is the two-dimensional, on-lattice, ballistic deposition model. It is "two-dimensional" because the profile of the surface is considered; "on-lattice" because falling and deposited particles are constrained to exist on a grid's cells; and "ballistic" because the falling particles follow linear, vertical trajectories.

What has been found (e.g., see Family and Vicsek, 1985) is that the internal structure of the aggregate is rather porous and homogeneous. And, on the other hand, the active zone exhibits interesting scaling behavior. Specifically, its width, defined as the standard deviation of the height, increases initially fast, but reaches a plateau as the number of deposited particles becomes large (in respect to the horizontal length of the grid). Additionally, it was shown that the active zone is not a fractal (e.g., in Meakin et al., 1986).  


## HOW IT WORKS


Resembling the DLA Alternate Linear model found at the NetLogo Library (Wilensky, 2005), this variant simulates on-lattice particles that fall from random positions, and stick to the surface's active zone. The active zone is delimited by the neighborhood of the surface's occupied sites. In the present, you can choose if a particle can stick: either to a nearest neighbor of a surface site, i.e., at its right, left or above neighboring unoccupied sites; or to the diagonally above left or right (next-nearest), or above unoccupied sites. Periodic boundary conditions are at play, i.e., the world is wrapped at its horizontal dimension. 

Initially, an even coat of particles is applied to the surface. At each time step, a new particle is created at a random position, and begins its fall. If it enters an active zone site, it sticks, and becomes part of the aggregate.


## HOW TO USE IT


Buttons:

- SETUP: Sets up the model.

- GO: Runs the model. While running, you can press it again to pause the simulation.


Swithes:

- SHOOT-NEAR: If "Off", the particles start their fall from the ceiling of the lattice. While "On", they start from the maximum height of the active zone, and the simulation speeds up. You can use this switch while the model is running.

- HIDDEN-ZONE: If "Off", you are able to see the active zone sites as red disks. Again, to speed up the simulation, turn this switch "On", and the active zone is not displayed.

- NEXT-NEAREST: When the switch is turned "Off", a particle can be deposited only to the nearest neighbors of a surface site. If "On", they can stick to the next-nearest neighbors additionally. You can use the switch during a simulation.


Plots:

- AVG. HEIGHT OF ACTIVE ZONE: Plots the average height of the active zone as particles are deposited.

- WIDTH OF ACTIVE ZONE: Plots the width versus the number of deposited particles.

- STEP SIZES OF ACTIVE ZONE: An histogram that illustrates the frequency distribution of the active zone's step sizes. Step size is defined as the absolute height difference of the active zone between two neighboring columns. 


Monitors:

- DENSITY: It monitors the density of the aggregate as time goes on. It is defined as the number of deposited particles, divided by the total number of sites belonging to the aggregate's volume.


## THINGS TO NOTICE


The average height of the active zone exhibits a positive linear relationship with the number of deposited particles. This is an indication of the uniform, non-fractal structure of the ballistic aggregate.

On the contrary, width does not grow continuously, but it tends to a limit size, specific to the horizontal length of the grid. 

The distribution of step sizes decays exponential-like. Most of the neighboring sites of the active zone are on the same height, or differ by only one lattice unit. And larger step sizes are rare, as their frequency drops dramatically. If the active zone was a fractal, large step sizes would be more frequent.


## THINGS TO TRY


Experiment with different horizontal to vertical dimension ratios. How do they affect the behavior of the width, or other quantities?

Find differences, in the quantities measured, between the two neighborhood rules. Which is the most profound change? For a more direct comparison, you can switch between the two rules while the model is running.


## EXTENDING THE MODEL


You can modify some of the rules of the present model. For example, you could build the "single step" ballistic-deposition model found in Meakin et al. (1986): "In this model two sites are added at a randomly selected active site, defined such that the height of the deposit is greater at the two neighboring positions". 

You could monitor other quantities, such as the total length of the active zone.

Instead of extending, you could simplify the code, to make the simulations run faster. The ballistic deposition model is characterized by its simplicity, since one has only to keep track of the active zone's height, and at each deposition event a maximum of three active zone heights have to be updated. Thus, the model is not computationally intensive at all. The present NetLogo version was built for the illustration of the model's mechanisms in a course presentation. That is the reason for the simulation of the active zone by turtles, and their display as red disks. To make it faster, e.g., you could remove the incorporation of the breed "active sites".



## RELATED MODELS


Uri Wilensky's family of DLA models, in the NetLogo Library.


## CREDITS AND REFERENCES


Family F. and Vicsek T. (1985) Scaling of the active zone in the Eden process on percolation networks and the ballistic deposition model. J. Phys. A 18, L75.

Meakin P., Ramanlal P., Sander L. M. and Ball R. C. (1986) Ballistic deposition on surfaces. Phys. Rev. A 34(6): 5091-5103.

Wilensky U. (2005) NetLogo DLA Alternate Linear model.
http://ccl.northwestern.edu/netlogo/models/DLAAlternateLinear. Center for Connected Learning and Computer-Based Modeling, Northwestern Institute on Complex Systems, Northwestern University, Evanston, IL.


## HOW TO CITE


If you mention this model in a publication, we ask that you include these citations for the model itself and for the NetLogo software:

* Kiziridis, D. (2014).  NetLogo Ballistic Deposition On-Lattice model. Institute for Cross-Disciplinary Physics and Complex Systems, IFISC (CSIC-UIB), Campus Universitat Illes Balears, Palma de Mallorca, Spain.
* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern Institute on Complex Systems, Northwestern University, Evanston, IL.


## COPYRIGHT AND LICENSE


Copyright 2014 Danis Kiziridis.

![CC BY-NC-SA 3.0](http://i.creativecommons.org/l/by-nc-sa/3.0/88x31.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
setup repeat 1500 [ go ]
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
